<?php

declare(strict_types=1);

namespace Erlage\Photogram\Tools\Fso\Disk;

use Erlage\Photogram\Settings;
use Aws\S3\S3Client as S3client;
use Erlage\Photogram\Constants\ServerConstants;

class DiskS3 implements InterfaceDisk
{
    /**
     * @var S3client
     */
    private $client;

    public function __construct()
    {
        $this -> client = new S3Client(
            array(
                'debug'                   => false,
                'use_path_style_endpoint' => true,
                'version'                 => 'latest',
                'region'                  => Settings::getString(ServerConstants::SS_TEXT_AWS_S3_REGION),

                'credentials'             => array(
                    'key'    => Settings::getString(ServerConstants::SS_TEXT_AWS_S3_KEY),
                    'secret' => Settings::getString(ServerConstants::SS_TEXT_AWS_S3_SECRET),
                ),
            )
        );
    }

    public function write(string $objectId, string $contents): void
    {
        $this -> client -> putObject(
            array(
                'ACL'    => 'public-read',
                'Body'   => $contents,
                'Key'    => $objectId,
                'Bucket' => Settings::getString(ServerConstants::SS_TEXT_AWS_S3_BUCKET),
            )
        );
    }

    public function read(string $objectId): string
    {
        $result = $this -> client -> getObject(
            array(
                'Bucket' => Settings::getString(ServerConstants::SS_TEXT_AWS_S3_BUCKET),
                'Key'    => $objectId,
            )
        );

        return $result -> body;
    }

    public function delete(string $objectId): void
    {
        $this -> client -> deleteObject(
            array(
                'Bucket' => Settings::getString(ServerConstants::SS_TEXT_AWS_S3_BUCKET),
                'Key'    => $objectId,
            )
        );
    }

    public function publicUrl(string $objectId): string
    {
        $publicUrl = $this -> client -> getObjectUrl(
            Settings::getString(ServerConstants::SS_TEXT_AWS_S3_BUCKET),
            $objectId,
        );

        return $publicUrl;
    }
}
